classdef MyWorLD < rl.env.MATLABEnvironment

    properties
        Map                                                               % 200*200*20 occupancy grid for buildings
        Drone = struct('x', 0, 'y', 0, 'z', 0)
        Goal = struct('x', 0, 'y', 0, 'z', 0)
        State = zeros(22,1)
        TrainingHistory = {}                                              % to store the states during training for visualization later
        NoiseLevel                                                        % Overal noise received at the ground
        GridDensity                                                       % density of map (200*200 grid)
    end

    properties (Access = private)
        mapSize = [200, 200, 20]
        TurinMaps                                                         % to store all the maps
        DensityMaps                                                       % to store all the density grids
        Distance2Goal                                                     % distance to target
        CurrentPointDensity
        CurrentNoise                                                      % the noise value at current step
        PreviousActions
        PreviousDronePosition
        CumulativeNoise
        optimalPathDeviation
    end

    properties (Access = private)
        NoiseRays                                                         % to store the direction vector of the rays emitted from noise source
        MapIndex
        LastEpisodeIndex
        StepCount
        Optimal_Path                                                      % optimal path found by A_star based on density and distance only
    end

    properties (Access = protected)                                       % Initialize internal flag to indicate episode termination
        IsDone = false
    end



    methods

        function this = MyWorLD()

            observationInfo = rlNumericSpec([22, 1]);
            observationInfo.Name = 'Observation';
            observationInfo.Description = 'Drone x, Drone y, Goal x, Goal y, Distance to goal, angle to goal, Sound Pressure Level at ground,  direction and angle to 3 closest and 3 furthest  buildings within 30 grid cell radius';

            actionInfo = rlNumericSpec([2 1]);
            actionInfo.LowerLimit = [-3; -3];
            actionInfo.UpperLimit = [3; 3];
            actionInfo.Name = 'Action';
            actionInfo.Description = 'Action';

            this = this@rl.env.MATLABEnvironment(observationInfo, actionInfo);

            aa = load('RayTracingRays.mat');                                        % load the generated noise rays
            bb = load('DensityMaps.mat');
            cc = load('TurinMaps.mat');

            this.TurinMaps = cc;                                          % store the maps in the properties
            this.DensityMaps = bb;

            rays = aa.r;
            this.NoiseRays = rays;                                        % store it in properties to be used in ray tracing function


            reset(this);                                                  % initialize reset function
        end





        function createMap(this)

            rows = 19;                                                    % Define the size of Maps cell array
            cols = 24;

            randomRow = randi(rows);                                      % Generate random row and column indices
            randomCol = randi(cols);

            this.MapIndex = [randomRow, randomCol];

            this.Map = this.TurinMaps.TurinMaps{randomRow, randomCol};
            this.GridDensity = this.DensityMaps.DensityMaps{randomRow, randomCol};

        end





        function state = reset(this)                                      % Start new episode

            createMap(this);

            random_number = randi([1, 2]);
            random_number2 = randi([1, 2]);

            if random_number == 1
                this.Drone.x = 1;
            else
                this.Drone.x = 200;
            end


            if random_number2 == 1
                this.Drone.y = 1;
            else
                this.Drone.y = 200;
            end

            this.Drone.z = 20;

            this.Goal.x = randi(this.mapSize(1));
            this.Goal.y = randi(this.mapSize(2));
            this.Goal.z = 20;

            this.Distance2Goal = zeros(2,1);
            this.CurrentNoise = zeros(2,1);
            this.PreviousActions = zeros(2,2);
            this.CumulativeNoise = [];
            this.CurrentPointDensity = zeros(2,1);
            %this.optimalPathDeviation = zeros(2,1);

            current_x = round(this.Drone.x);
            current_y = round(this.Drone.y);

            current_density = this.GridDensity(current_y, current_x);

            [~, normalized_angle] = this.findDensity(current_x, current_y);
            this.CurrentPointDensity(1,1) = current_density;

            % GoalRegister = int8(zeros(this.mapSize(1), this.mapSize(2)));
            % GoalRegister(this.Goal.y, this.Goal.x) = 1;
            %
            % this.Optimal_Path = this.A_star(current_x, current_y, this.GridDensity, GoalRegister, 10);
            % distance2optimalpath = this.distance_of_Drone_to_optimalPath();
            %
            % this.optimalPathDeviation(1,1) = distance2optimalpath;

            [coord, spl, NoiseValue] = this.RayTracing();
            Building_directions = this.findClosestBuildings();

            this.CurrentNoise(1,1) = NoiseValue;                          % noise value at current step
            this.Distance2Goal(1,1) = norm([this.Drone.x, this.Drone.y] - [this.Goal.x, this.Goal.y]);

            direction2goal = [this.Goal.x,  this.Goal.y] - [ this.Drone.x, this.Drone.y];  % get the direction to target from the drone location
            angle = atan2d(direction2goal(2), direction2goal(1));
            normalized_direction_angle = (angle + 180) / 360;

            epsilon = 1e-4;
            normalized_direction_angle = normalized_direction_angle * (1 - 2 * epsilon) + epsilon;

            state_unnormalized = [this.Drone.x; this.Drone.y; this.Goal.x; this.Goal.y; this.Distance2Goal(1,1); normalized_direction_angle; NoiseValue; Building_directions; current_density; normalized_angle];  % current step observation
            state = this.normalizeObservation(state_unnormalized);
            this.State = state_unnormalized;

            this.StepCount = 0;
        end





        function [nextobs,reward,isdone,loggedSignals] = step(this,action) % Step the environment to the next state, given the action

            arguments
                this
                action {mustBeNumeric, mustBeFinite, mustBeReal, mustBeVector}
            end


            this.PreviousDronePosition = [this.Drone.x, this.Drone.y];
            this.StepCount = this.StepCount + 1;

            clippedAction = max(min(action, [3 ; 3]), [-3; -3]);
            new_position = [this.Drone.x, this.Drone.y] + clippedAction';


            if all(new_position >= 1) && all(new_position <= this.mapSize(1:2))  % Check if the new position is within the map boundaries

                this.Drone.x = new_position(1);
                this.Drone.y = new_position(2);
            end


            this.PreviousActions = [clippedAction';  this.PreviousActions(1,:)];


            current_x = round(this.Drone.x);
            current_y = round(this.Drone.y);

            current_density = this.GridDensity(current_y, current_x);
            this.CurrentPointDensity = [current_density; this.CurrentPointDensity(1)];

            [~, normalized_angle] = this.findDensity(current_x, current_y);

            % distance2optimalpath = this.distance_of_Drone_to_optimalPath();


            [coord, spl, NoiseValue] = this.RayTracing();
            Building_directions = this.findClosestBuildings();

            newDistance = norm([this.Drone.x, this.Drone.y] - [this.Goal.x, this.Goal.y]);

            this.Distance2Goal = [newDistance; this.Distance2Goal(1)];
            this.CurrentNoise = [NoiseValue;  this.CurrentNoise(1)];
            this.CumulativeNoise = [NoiseValue; this.CumulativeNoise];

            direction2goal = [this.Goal.x,  this.Goal.y] - [ this.Drone.x, this.Drone.y];
            angle = atan2d(direction2goal(2), direction2goal(1));
            normalized_direction_angle = (angle + 180) / 360;

            epsilon = 1e-4;
            normalized_direction_angle = normalized_direction_angle * (1 - 2 * epsilon) + epsilon;


            nextobs_unnormalized = [this.Drone.x; this.Drone.y; this.Goal.x; this.Goal.y; newDistance; normalized_direction_angle; NoiseValue; Building_directions; current_density; normalized_angle]; % next step observation

            nextobs = this.normalizeObservation(nextobs_unnormalized);
            reward = this.calculateReward(nextobs_unnormalized);
            isdone = this.isDone(nextobs_unnormalized);
            this.State = nextobs_unnormalized;
            this.IsDone = isdone;
            loggedSignals = [];


            if isempty(this.TrainingHistory)                            % for training visualization

                history{1,1} = nextobs_unnormalized;
                history{2,1} = this.MapIndex;
                history{3,1} = this.Optimal_Path;
                history{4,1} = coord;                                     % coordinate of each noise ray as it travells
                history{5,1} = spl;                                       % noise level of ray along the path
                this.LastEpisodeIndex = this.MapIndex;

            elseif isequal(this.LastEpisodeIndex, this.MapIndex)

                history{1,1} = nextobs_unnormalized;
                history{2,1} = [];
                history{3,1} = [];
                history{4,1} = coord;
                history{5,1} = spl;

            elseif ~isequal(this.LastEpisodeIndex, this.MapIndex)

                history{1,1} = nextobs_unnormalized;
                history{2,1} = this.MapIndex;
                history{3,1} = this.Optimal_Path;
                history{4,1} = coord;
                history{5,1} = spl;
                this.LastEpisodeIndex = this.MapIndex;
            end

            this.TrainingHistory = [this.TrainingHistory, history];       % Store the current state and the building map and density map in the training history to visualize later

        end





        function reward = calculateReward(this, nextobs)

            idlePenalty = 0;
            td_regularization_penalty = 0;


            if this.isDone(nextobs)
                reward = 2;                                               % reward for reaching the goal.
                return
            end


            if this.State(1:2) == nextobs(1:2)
                idlePenalty = 1;
            end


            if this.StepCount > 1

                prev_movement1 = this.PreviousActions(1,:);
                prev_movement2 = this.PreviousActions(2,:);
                angle1 = atan2(prev_movement1(2), prev_movement1(1));
                angle2 = atan2(prev_movement2(2), prev_movement2(1));
                angle_diff = mod(angle2 - angle1 + pi, 2*pi) - pi;
                angle_diff_degrees = rad2deg(angle_diff);
                td_regularization_penalty = abs(angle_diff_degrees)/180;
            end


            distance_reduction = this.Distance2Goal(2) - this.Distance2Goal(1);    % Calculate distance reduction towards the goal


            if distance_reduction > 3

                DistanceEfficiency = 1;
            elseif distance_reduction < -3

                DistanceEfficiency = - 1;
            else
                DistanceEfficiency = distance_reduction / 3;
            end


            DistancePenalty =  - 0.5 * (DistanceEfficiency - 1);


            total_Noise = sum(this.CumulativeNoise);
            max_Noise = 280 * 66.12;

            cumulative_noise_penalty = total_Noise / max_Noise;        



            action = norm(this.PreviousActions(1,:));
            threshold = 0.2358 * action^2 + 4.819 * action + 30.8;
            threshold = floor(threshold);

            if this.CurrentNoise(1) > threshold

                NoisePenalty = 1;
            else
                difference = threshold - this.CurrentNoise(1);
                NoisePenalty =  exp(-3 * (difference / threshold));
            end


            density_change = this.CurrentPointDensity(1) - this.CurrentPointDensity(2);

            if density_change > 0

                PopulationDensityPenalty = 1;

            else
                PopulationDensityPenalty = 0.1;
            end


            reward = - 0.02*DistancePenalty  - 0.07*PopulationDensityPenalty - 0.01*cumulative_noise_penalty...  % reward function
                - 0.43*NoisePenalty - 0.47*td_regularization_penalty - idlePenalty;

        end





        function done = isDone(~, nextobs)                                % Check if the episode is done based on the current location of drone and target

            done = sqrt((nextobs(1) - nextobs(3))^2 + (nextobs(2) - nextobs(4))^2) <= 1.4143; % isdone when drone closer than radius of diagonal of 1 grid cell to target
        end





        function OptimalPath = A_star(~, StartX, StartY, MAP, GoalRegister, Connecting_Distance)

            % FINDING A_STAR PATH IN A WEIGHTED GRID without obstacles (based on distance and population density)

            %Generating goal nodes, which is represented by a matrix. Several goals can be speciefied, in which case the pathfinder will find the closest goal.
            %a cell with the value 1 represent a goal cell

            %Number of Neighboors one wants to investigate from each cell. A larger
            %number of nodes means that the path can be alligned in more directions.


            % Preallocation of Matrices
            [Height, Width] = size(MAP); % Height and width of matrix
            GScore = zeros(Height, Width);           % Matrix keeping track of G-scores
            FScore = single(inf(Height, Width));     % Matrix keeping track of F-scores (only open list)
            Hn = single(zeros(Height, Width));       % Heuristic matrix
            OpenMAT = int8(zeros(Height, Width));    % Matrix keeping of open grid cells
            ClosedMAT = int8(zeros(Height, Width));  % Matrix keeping track of closed grid cells
            ParentX = int16(zeros(Height, Width));   % Matrix keeping track of X position of parent
            ParentY = int16(zeros(Height, Width));   % Matrix keeping track of Y position of parent


            %%% Setting up matrices representing neighbours to be investigated
            NeighbourCheck = ones(2 * Connecting_Distance + 1);
            Dummy = 2 * Connecting_Distance + 2;
            Mid = Connecting_Distance + 1;
            for i = 1:Connecting_Distance - 1
                NeighbourCheck(i, i) = 0;
                NeighbourCheck(Dummy - i, i) = 0;
                NeighbourCheck(i, Dummy - i) = 0;
                NeighbourCheck(Dummy - i, Dummy - i) = 0;
                NeighbourCheck(Mid, i) = 0;
                NeighbourCheck(Mid, Dummy - i) = 0;
                NeighbourCheck(i, Mid) = 0;
                NeighbourCheck(Dummy - i, Mid) = 0;
            end
            NeighbourCheck(Mid, Mid) = 0;
            [row, col] = find(NeighbourCheck == 1);
            Neighbours = [row col] - (Connecting_Distance + 1);
            N_Neighbours = size(col, 1);
            %%% End of setting up matrices representing neighbours to be investigated

            %%%%%%%%% Creating Heuristic-matrix based on value of each node
            for k = 1:size(GoalRegister, 1)
                for j = 1:size(GoalRegister, 2)
                    if MAP(k, j) > 0  % Check if the node value is greater than 0

                        distance_to_goal = (sqrt((StartX - j)^2 + (StartY - k)^2)) / norm([200,200]); % Heuristic is based on the density value of the node (between 0 and 1) and normalized distance to target (between 0 and 1), with 60-40% for the density
                        Hn(k, j) = 0.6*MAP(k, j) + 0.4*distance_to_goal;
                    end
                end
            end
            % End of creating Heuristic-matrix.

            % Initialising start node with FValue and opening first node.
            FScore(StartY, StartX) = Hn(StartY, StartX);
            OpenMAT(StartY, StartX) = 1;

            while true % Code will break when path found or when no path exists
                MINopenFSCORE = min(min(FScore));
                if MINopenFSCORE == inf
                    % Failure!
                    OptimalPath = inf;
                    RECONSTRUCTPATH = 0;
                    break
                end
                [CurrentY, CurrentX] = find(FScore == MINopenFSCORE, 1, 'first');

                if GoalRegister(CurrentY, CurrentX) == 1
                    % Goal reached!
                    RECONSTRUCTPATH = 1;
                    break
                end

                % Removing node from OpenList to ClosedList
                OpenMAT(CurrentY, CurrentX) = 0;
                FScore(CurrentY, CurrentX) = inf;
                ClosedMAT(CurrentY, CurrentX) = 1;

                for p = 1:N_Neighbours
                    i = Neighbours(p, 1); % Y
                    j = Neighbours(p, 2); % X
                    if CurrentY + i < 1 || CurrentY + i > Height || CurrentX + j < 1 || CurrentX + j > Width
                        continue
                    end

                    if ClosedMAT(CurrentY + i, CurrentX + j) == 0 % Neighbour is open
                        % Calculate the tentative gScore based on the Euclidean distance and population density
                        tentative_gScore = GScore(CurrentY, CurrentX) + 0.4*(sqrt(i^2 + j^2))/(norm([200,200])) + 0.6*MAP(CurrentY + i, CurrentX + j);

                        if tentative_gScore < GScore(CurrentY + i, CurrentX + j) || OpenMAT(CurrentY + i, CurrentX + j) == 0
                            ParentX(CurrentY + i, CurrentX + j) = CurrentX;
                            ParentY(CurrentY + i, CurrentX + j) = CurrentY;
                            GScore(CurrentY + i, CurrentX + j) = tentative_gScore;
                            FScore(CurrentY + i, CurrentX + j) = tentative_gScore + Hn(CurrentY + i, CurrentX + j);
                            OpenMAT(CurrentY + i, CurrentX + j) = 1;
                        end
                    end
                end
            end

            k = 2;
            if RECONSTRUCTPATH
                OptimalPath(1, :) = [CurrentY CurrentX];
                while RECONSTRUCTPATH
                    CurrentXDummy = ParentX(CurrentY, CurrentX);
                    CurrentY = ParentY(CurrentY, CurrentX);
                    CurrentX = CurrentXDummy;
                    OptimalPath(k, :) = [CurrentY CurrentX];
                    k = k + 1;
                    if CurrentX == StartX && CurrentY == StartY
                        break
                    end
                end
            end
        end





        function closest_distance = distance_of_Drone_to_optimalPath(this)

            path = this.Optimal_Path;
            path = flipud(path);
            path = path(:, [2, 1]);
            currentPosition = [this.Drone.x, this.Drone.y];

            segments = [path(1:end-1, :), path(2:end, :)];

            point_vectors = bsxfun(@minus, currentPosition, segments(:, 1:2));  % Vector from line segment start to drone
            segment_vectors = segments(:, 3:4) - segments(:, 1:2);        % Vector from line segment start to end

            t = dot(point_vectors, segment_vectors, 2) ./ dot(segment_vectors, segment_vectors, 2);% Project point vectors onto segment vectors
            t = max(0, min(1, t));                                        % Ensure projection falls within segment bounds

            closest_points = segments(:, 1:2) + t .* segment_vectors;

            distances = sqrt(sum((currentPosition - closest_points).^2, 2));
            closest_distance = min(distances);

        end





        function [Surrounding_density, normalized_angle] = findDensity(this, current_x, current_y)  %calculating surrounding density of drone and angle to max density

            sizes = [15, 51];                                             % defining 2 subgrids around drone
            direction_vector = zeros(2,2);
            angle_degrees = zeros(2,1);
            normalized_angle = zeros(2,1);
            Surrounding_density = zeros(8,1);

            for i=1:2

                half_size = (sizes(i) - 1) / 2;
                grid_center = ceil(sizes(i) / 2);

                row_indices = max(1, min(size(this.mapSize, 1), current_y - half_size : current_y + half_size));
                col_indices = max(1, min(size(this.mapSize, 2), current_x - half_size : current_x + half_size));

                subgrid_density = this.GridDensity(row_indices,col_indices);

                max_density = max(subgrid_density, [], 'all');            % max density in the subgrid

                [row_indices2, col_indices2] = find(subgrid_density == max_density);

                distances = sqrt((row_indices2 - grid_center).^2 + (col_indices2 - grid_center).^2);
                [~, closest_index] = min(distances);
                maxdensity_barycenter = [row_indices2(closest_index), col_indices2(closest_index)]; % Choose the closest point as the representative point

                direction_vector(i,:) = maxdensity_barycenter - [grid_center, grid_center];  % get the direction to barycenter from the center of subgrid(drone)
                angle_degrees(i) = atan2d(direction_vector(i,2), direction_vector(i,1));


                if all(direction_vector(i,:) == 0)                        % if density in subgrid is all same, we find the angle to the target otherwise get the angle to min density in subgrid

                    direction = [this.Goal.x,  this.Goal.y] - [ this.Drone.x, this.Drone.y];  % get the direction to barycenter from the center of subgrid(drone)
                    angle = atan2d(direction(2), direction(1));
                    normalized_angle(i) = (angle + 180) / 360;

                    normalized_angle(i) =  mod(normalized_angle(i) + 0.5, 1);
                else
                    normalized_angle(i) = (angle_degrees(i) + 180) / 360;
                end
            end

            epsilon = 1e-4;
            normalized_angle = normalized_angle * (1 - 2 * epsilon) + epsilon;

        end





        function Building_directions = findClosestBuildings(this)


            scanRadius = 30;
            obstacles = [];
            [dimX, dimY, ~] = size(this.Map);                             % Extracting the size of the map

            currentPosition = [round(this.Drone.x), round(this.Drone.y), round(this.Drone.z)];
            target_point  = [this.Goal.x, this.Goal.y, this.Goal.z];

            direction_vector = [target_point(1) - currentPosition(1), target_point(2) - currentPosition(2)]; % Calculate angle between current point and target point
            angle = atan2(direction_vector(2), direction_vector(1));

            angle_plus_40 = angle + deg2rad(40);
            angle_minus_40 = angle - deg2rad(40);

            % Scan the map within the specified radius
            for x = max(1, currentPosition(1) - scanRadius) : min(dimX, currentPosition(1) + scanRadius)
                for y = max(1, currentPosition(2) - scanRadius) : min(dimY, currentPosition(2) + scanRadius)
                    % Check if point (x, y) is inside the circle
                    if ((x - currentPosition(1))^2 + (y - currentPosition(2))^2 <= scanRadius^2)
                        % Check if point (x, y) is between the two lines(-40 and +40)
                        angle_point = atan2(y - currentPosition(2), x - currentPosition(1));
                        if (angle_point >= angle_minus_40 && angle_point <= angle_plus_40)
                            if this.Map(y, x, 1) == 1
                                obstaclePoint = [x, y, 1];
                                distance = norm(currentPosition - obstaclePoint);
                                obstacles = [obstacles; distance, obstaclePoint];  % if the point is occupied, store its coordinates and its distance to drone
                            end
                        end
                    end
                end
            end



            Building_directions = zeros(12, 1);
            close_buildings = zeros(6, 1);
            far_buildings = zeros(6, 1);

            if ~isempty(obstacles)

                [~, sortedIdx] = sort(obstacles(:,1));                    % Sort the found obstacles by distance
                sortedObstacles = obstacles(sortedIdx, :);

                numObstacles = min(size(sortedObstacles, 1), 3);

                % Extract coordinates and distances for the closest obstacles
                closestObstaclesPosition = sortedObstacles(1:numObstacles, 2:end);
                closestObstaclesDistance = sortedObstacles(1:numObstacles, 1);

                % Extract coordinates and distances for the furthest obstacles
                furthestObstaclesPosition = sortedObstacles(end-numObstacles+1:end, 2:end);
                furthestObstaclesDistance = sortedObstacles(end-numObstacles+1:end, 1);

                normalized_closest_distances = closestObstaclesDistance / norm([dimX, dimY] - [1,1]);
                normalized_furthest_distances = furthestObstaclesDistance / norm([dimX, dimY] - [1,1]);

                closest_direction_vectors = closestObstaclesPosition - repmat(currentPosition, numObstacles, 1);
                furthest_direction_vectors = furthestObstaclesPosition - repmat(currentPosition, numObstacles, 1);

                closest_angle_degrees = atan2d(closest_direction_vectors(:,2), closest_direction_vectors(:,1));
                furthest_angle_degrees = atan2d(furthest_direction_vectors(:,2), furthest_direction_vectors(:,1));

                closest_normalized_angle = (closest_angle_degrees + 180) / 360;
                furthest_normalized_angle = (furthest_angle_degrees + 180) / 360;

                epsilon = 1e-4;                                          % Adjust to exclude 0 and 1

                closest_tempDirections = closest_normalized_angle * (1 - 2 * epsilon) + epsilon;
                furthest_tempDirections = furthest_normalized_angle * (1 - 2 * epsilon) + epsilon;

                % Replace 0.5 with 1 if the corresponding obstacle position is the same as the current position for the closest obstacles
                for i = 1:numObstacles
                    if closest_tempDirections(i) == 0.5 && isequal(closestObstaclesPosition(i,1:2), currentPosition(1,1:2))
                        closest_tempDirections(i) = 1;
                    end

                    if furthest_tempDirections(i) == 0.5 && isequal(furthestObstaclesPosition(i,1:2), currentPosition(1,1:2))
                        furthest_tempDirections(i) = 1;
                    end
                end


                % Store directions and distances alternately
                for i = 1:numObstacles
                    close_buildings(2*i - 1) = closest_tempDirections(i);
                    close_buildings(2*i) = normalized_closest_distances(i);

                    far_buildings(2*i - 1) = furthest_tempDirections(i);
                    far_buildings(2*i) = normalized_furthest_distances(i);
                end

                Building_directions = [close_buildings; far_buildings];

            end
        end





        function normalized_observation = normalizeObservation(this, state)

            drone_position = state(1:2);
            goal_position = state(3:4);
            distance2target = state(5);
            noiseLevel = state(7);

            normalized_drone = drone_position ./ [this.mapSize(1), this.mapSize(2)]';
            normalized_goal = goal_position ./ [this.mapSize(1), this.mapSize(2)]';
            normalized_noise = noiseLevel / 100;
            normalized_distance = distance2target / norm([this.mapSize(1),this.mapSize(1)] - [1,1]);

            normalized_observation = [normalized_drone; normalized_goal; normalized_distance; state(6); normalized_noise; state(8:22)];

        end





        function [impactCoord, SPL, Noise] = RayTracing(this)


            if all(this.PreviousActions(1,:) == 0)

                velocity = norm([3,3]) / 0.3 * 3.6 * 2;                   % distance travelled divided by sample time to get speed in (m/s) then multiply by 3.6 to get (km/h) then multiply by 2 since grid resolution is 2 meter per cell
            else
                velocity = norm(this.PreviousActions(1,:)) / 0.3 * 3.6 * 2;
            end


            NoiseSource = 60.24 * exp(0.003379 * velocity);


            N = 46;
            maxDistance = 200;
            absorptionCoeffcient = 0.00076 * 2;
            stepLength = 0.3;

            rays = this.NoiseRays;

            RayDistance = cell(N, 1);                                     % Store distances of each segment of each ray  while travelling
            impactCoord = cell(N, 1);                                     % Store ccoordinates of each segment of each ray  while travelling
            distLoss = cell(N, 1);                                        % sound pressure level loss at each segment of ray path
            SPL = cell(N, 1);                                             % Sound pressure level at each point of the rays
            ReceiverCells = zeros(N, 1);                                  % Noise received at ground from each ray

            for iRay = 1:N

                currentX = this.Drone.x;
                currentY = this.Drone.y;
                currentZ = this.Drone.z;

                xDir = rays(iRay,1);
                yDir = rays(iRay,2);
                zDir = rays(iRay,3);

                ray_distance = 0;
                counter = 0;

                RayDistance{iRay} = [];
                impactCoord{iRay} = [];
                distLoss{iRay} = [];
                SPL{iRay} = [];

                while ray_distance <= maxDistance

                    currentX = currentX + xDir * stepLength;
                    currentY = currentY + yDir * stepLength;
                    currentZ = currentZ + zDir * stepLength;

                    currentX = max(1, min(currentX, this.mapSize(1)));
                    currentY = max(1, min(currentY, this.mapSize(2)));
                    currentZ = max(1, min(currentZ, this.mapSize(3)));

                    counter = counter + 1;

                    if currentX == 1 || currentX == this.mapSize(1) || currentY == 1 || currentY == this.mapSize(2) ||  currentZ == this.mapSize(3)  % if the ray reaches the map boundary (except the ground), ray tracing is terminated for the current ray and we go to the next ray.  except when hitting obstacles where its reflected

                        impactCoord{iRay}(counter,:) = [currentX, currentY, currentZ];

                        if size(impactCoord{iRay}, 1) == 1
                            distance = sqrt((currentX -  this.Drone.x)^2 + (currentY -  this.Drone.y)^2 + (currentZ -  this.Drone.z)^2);
                        else
                            distance = sqrt((currentX -  impactCoord{iRay}(end-1,1))^2 + (currentY -  impactCoord{iRay}(end-1,2))^2 + (currentZ -  impactCoord{iRay}(end-1,3))^2);
                        end

                        RayDistance{iRay}(counter) =  distance;


                        attenuation = distance * absorptionCoeffcient;    % loss due to atmospheric absorption

                        if sum(RayDistance{iRay}) >= 1

                            inverseLaw = 20 * log10(sum(RayDistance{iRay}) * 2) + 11;
                        else
                            inverseLaw = 0;
                        end


                        distLoss{iRay}(counter) = attenuation;
                        splValue = NoiseSource - sum(distLoss{iRay}) - inverseLaw;

                        if splValue <= 0
                            break
                        end


                        SPL{iRay}(counter,1) = splValue;
                        ray_distance = ray_distance + distance;

                        if round(currentZ) <= 2 && ReceiverCells(iRay) == 0
                            ReceiverCells(iRay,1) = SPL{iRay}(counter,1);
                        end

                        break;
                    end



                    if this.Map(ceil(currentY), ceil(currentX), ceil(currentZ)) == 1 || this.Map(floor(currentY), floor(currentX), floor(currentZ)) == 1 || currentZ == 1 || ...
                            (this.Map(ceil(currentY), ceil(currentX), min(ceil(currentZ) + 1, size(this.Map, 3))) == 1 && this.Map(ceil(currentY), ceil(currentX), max(ceil(currentZ) - 1, 1)) == 1) || ...
                            (this.Map(floor(currentY), floor(currentX), min(floor(currentZ) + 1, size(this.Map, 3))) == 1 && this.Map(floor(currentY), floor(currentX), max(floor(currentZ) - 1, 1)) == 1)

                        [hit, hitFace] = this.checkHitFace(currentX, currentY, currentZ);  % get the which face was hit

                        if hit                                            % Reflect the ray based on the hit face
                            normalVectors = [
                                1, 0, 0;
                                -1, 0, 0;
                                0, 1, 0;
                                0,-1, 0;
                                0, 0, 1;
                                0, 0,-1
                                ];

                            hitNormal = normalVectors(hitFace, :);        % normal vector

                            [reflectedX, reflectedY, reflectedZ] = this.ReflectRay( xDir, yDir, zDir, hitNormal); % get the reflected directions
                            xDir = reflectedX;
                            yDir = reflectedY;
                            zDir = reflectedZ;

                            impactCoord{iRay}(counter,:) = [currentX, currentY, currentZ];

                            if size(impactCoord{iRay}, 1) == 1
                                distance = sqrt((currentX -  this.Drone.x)^2 + (currentY -  this.Drone.y)^2 + (currentZ -  this.Drone.z)^2);
                            else
                                distance = sqrt((currentX -  impactCoord{iRay}(end-1,1))^2 + (currentY -  impactCoord{iRay}(end-1,2))^2 + (currentZ -  impactCoord{iRay}(end-1,3))^2);
                            end

                            RayDistance{iRay}(counter) =  distance;

                            attenuation = distance * absorptionCoeffcient;

                            if sum(RayDistance{iRay}) >= 1

                                inverseLaw = 20 * log10(sum(RayDistance{iRay}) * 2) + 11;
                            else
                                inverseLaw = 0;
                            end


                            distLoss{iRay}(counter) = attenuation;
                            splValue = NoiseSource - sum(distLoss{iRay}) - inverseLaw;

                            if splValue <=0
                                break
                            end


                            SPL{iRay}(counter,1) = splValue;
                            ray_distance = ray_distance + distance;

                            if round(currentZ) <= 2 && ReceiverCells(iRay) == 0
                                ReceiverCells(iRay,1) = SPL{iRay}(counter,1);
                            end

                            continue;
                        end
                    end


                    impactCoord{iRay}(counter,:) = [currentX, currentY, currentZ];

                    if size(impactCoord{iRay}, 1) == 1
                        distance = sqrt((currentX -  this.Drone.x)^2 + (currentY -  this.Drone.y)^2 + (currentZ -  this.Drone.z)^2);
                    else
                        distance = sqrt((currentX -  impactCoord{iRay}(end-1,1))^2 + (currentY -  impactCoord{iRay}(end-1,2))^2 + (currentZ -  impactCoord{iRay}(end-1,3))^2);
                    end


                    RayDistance{iRay}(counter) =  distance;

                    attenuation = distance * absorptionCoeffcient;


                    if sum(RayDistance{iRay}) >= 1

                        inverseLaw = 20 * log10(sum(RayDistance{iRay}) * 2) + 11;
                    else
                        inverseLaw = 0;
                    end

                    distLoss{iRay}(counter) = attenuation;

                    splValue = NoiseSource - sum(distLoss{iRay}) - inverseLaw;

                    if splValue <=0
                        break
                    end

                    SPL{iRay}(counter,1) = splValue;

                    ray_distance = ray_distance + distance;

                    if round(currentZ) <= 2 && ReceiverCells(iRay) == 0
                        ReceiverCells(iRay,1) = SPL{iRay}(counter,1);
                    end

                end
            end



            nonZeroReceiverCells = ReceiverCells(ReceiverCells ~= 0);

            if ~isempty(nonZeroReceiverCells)

                Noise = 10 * log10(sum(10 .^ (nonZeroReceiverCells ./ 10)));
            else
                Noise = 0;
            end


            this.NoiseLevel = Noise;
        end





        function [xDir_r, yDir_r, zDir_r] = ReflectRay( ~, xDir, yDir, zDir, hitNormal)   % to reflect the ray when hitting obstacle

            normal_Vector = hitNormal;                                    % Use the provided hitNormal (normal to ray hit face)
            normalVector = normal_Vector / norm(normal_Vector);

            incidentVector = [xDir, yDir, zDir];                          % Use the reflection formula: R = I - 2 * dot(I, N) * N for specular reflection only(neglecting diffuse reflection)
            reflectionVector = incidentVector - 2 * dot(incidentVector, normalVector) * normalVector;

            reflectionVector = reflectionVector / norm(reflectionVector); % Normalize the reflection vector

            xDir_r = reflectionVector(1);
            yDir_r = reflectionVector(2);                                 % Reflected direction
            zDir_r = reflectionVector(3);
        end





        function [hit, hitFace] = checkHitFace(~, x, y, z)                % to check which face of obstacle was hit by ray

            epsilon = 0.1;

            hit = false;
            hitFace = 0;

            checkAxis = @(val, axis, dir) abs(val - round(val)) < epsilon && ((val > round(val)) == dir);

            if checkAxis(x, 'x', true)
                hit = true;
                hitFace = 1;
            elseif checkAxis(x, 'x', false)
                hit = true;
                hitFace = 2;
            elseif checkAxis(y, 'y', true)
                hit = true;
                hitFace = 3;
            elseif checkAxis(y, 'y', false)
                hit = true;
                hitFace = 4;
            elseif checkAxis(z, 'z', true)
                hit = true;
                hitFace = 5;
            elseif checkAxis(z, 'z', false)
                hit = true;
                hitFace = 6;
            end
        end


    end
end